import express from 'express';
import cors from 'cors';
import fetch from 'node-fetch';
import https from 'https';

// Configure HTTPS agent to bypass SSL verification for development
const httpsAgent = new https.Agent({
  rejectUnauthorized: false
});

const app = express();
const PORT = 8080;

// Configuration - point to your main server
// Use the current Replit backend URL
const BACKEND_URL = process.env.BACKEND_URL || 
                   'https://afc8f4e5-ffe6-415c-bd1e-9c62bd8b49b1-00-3adg58ea9ee6l.spock.replit.dev';

console.log(`🔗 PROXY SERVER: Starting proxy server on port ${PORT}`);
console.log(`🎯 BACKEND_URL: ${BACKEND_URL}`);
if (BACKEND_URL.startsWith('https')) {
  console.log(`🔒 SSL: Certificate validation disabled for development`);
}

// Screenshot request queue to handle Chrome API rate limits
class ScreenshotQueue {
  constructor() {
    this.queue = [];
    this.processing = false;
    this.lastRequest = 0;
    this.minInterval = 200; // 200ms between requests to respect Chrome rate limits
  }

  async add(requestData) {
    return new Promise((resolve, reject) => {
      this.queue.push({ requestData, resolve, reject });
      this.process();
    });
  }

  async process() {
    if (this.processing || this.queue.length === 0) return;
    
    this.processing = true;
    
    while (this.queue.length > 0) {
      const { requestData, resolve, reject } = this.queue.shift();
      
      try {
        // Respect rate limits
        const now = Date.now();
        const timeSinceLastRequest = now - this.lastRequest;
        if (timeSinceLastRequest < this.minInterval) {
          await new Promise(r => setTimeout(r, this.minInterval - timeSinceLastRequest));
        }
        
        const result = await this.forwardRequest(requestData);
        this.lastRequest = Date.now();
        resolve(result);
      } catch (error) {
        reject(error);
      }
    }
    
    this.processing = false;
  }

  async forwardRequest(requestData) {
    console.log(`📡 PROXY: Forwarding ${requestData.method} ${requestData.path}`);
    
    const response = await fetch(`${BACKEND_URL}${requestData.path}`, {
      method: requestData.method,
      headers: {
        'Content-Type': 'application/json',
        ...requestData.headers
      },
      body: requestData.body,
      agent: BACKEND_URL.startsWith('https') ? httpsAgent : undefined
    });

    const responseText = await response.text();
    console.log(`📊 PROXY: Queue response status: ${response.status}, Content-Length: ${response.headers.get('content-length')}, Body length: ${responseText.length}`);
    
    if (!responseText) {
      console.error(`❌ PROXY: Empty response from backend for ${requestData.path}`);
      throw new Error('Empty response from backend server');
    }

    const data = JSON.parse(responseText);
    console.log(`✅ PROXY: Response ${response.status} for ${requestData.path}`);
    return { status: response.status, data };
  }
}

const screenshotQueue = new ScreenshotQueue();

// Middleware
app.use(cors({
  origin: ['chrome-extension://*', 'moz-extension://*', 'http://localhost:*', 'https://localhost:*'],
  credentials: true
}));

app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Health check endpoint
app.get('/health', (req, res) => {
  console.log('🔍 PROXY: Health check requested');
  res.json({ 
    status: 'OK', 
    proxy: 'ActivityRecorder Proxy Server',
    backend: BACKEND_URL,
    timestamp: new Date().toISOString()
  });
});

// Queue-managed screenshot endpoint
app.post('/api/upload-screenshot', async (req, res) => {
  try {
    console.log(`📸 PROXY: Queuing screenshot upload request`);
    const result = await screenshotQueue.add({
      path: '/api/upload-screenshot',
      method: 'POST',
      headers: req.headers,
      body: JSON.stringify(req.body)
    });

    res.status(result.status).json(result.data);
  } catch (error) {
    console.error('❌ PROXY: Screenshot upload error:', error);
    res.status(500).json({ error: 'Proxy server error', details: error.message });
  }
});

// General proxy endpoint for all other API requests
app.all('/api/*', async (req, res) => {
  try {
    const fullUrl = `${BACKEND_URL}${req.path}`;
    console.log(`🔄 PROXY: ${req.method} ${req.path}`);
    console.log(`🌐 PROXY: Full URL: ${fullUrl}`);
    console.log(`📦 PROXY: Request body:`, req.body);
    
    const response = await fetch(fullUrl, {
      method: req.method,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'User-Agent': 'ActivityRecorder-Client/1.0'
      },
      body: req.method !== 'GET' ? JSON.stringify(req.body) : undefined,
      agent: BACKEND_URL.startsWith('https') ? httpsAgent : undefined
    });

    // Check if response has content before parsing JSON
    const responseText = await response.text();
    console.log(`📊 PROXY: Response status: ${response.status}, Content-Length: ${response.headers.get('content-length')}, Body length: ${responseText.length}`);
    console.log(`🔍 PROXY: Response URL: ${response.url}`);
    console.log(`🔍 PROXY: Response redirected: ${response.redirected}`);
    
    if (!responseText) {
      console.error(`❌ PROXY: Empty response from backend for ${req.path}`);
      console.error(`📊 PROXY: Response headers:`, [...response.headers.entries()]);
      return res.status(502).json({ error: 'Empty response from backend server' });
    }
    
    // Check if we got HTML instead of JSON (indicating a 404 or routing error)
    if (responseText.trim().startsWith('<!DOCTYPE html>')) {
      console.error(`🚨 PROXY: Received HTML page instead of JSON API response`);
      console.error(`🚨 PROXY: This suggests the URL is not reaching the correct API endpoint`);
      console.error(`🚨 PROXY: First 200 chars: ${responseText.substring(0, 200)}`);
      return res.status(502).json({ 
        error: 'Backend returned HTML page instead of JSON API response',
        hint: 'API endpoint may not be available or URL is incorrect'
      });
    }
    
    let data;
    try {
      data = JSON.parse(responseText);
    } catch (parseError) {
      console.error(`❌ PROXY: Invalid JSON from backend for ${req.path}:`, responseText);
      return res.status(502).json({ error: 'Invalid JSON response from backend', response: responseText });
    }
    
    res.status(response.status).json(data);
  } catch (error) {
    console.error(`❌ PROXY: Error forwarding ${req.path}:`, error);
    res.status(500).json({ error: 'Proxy server error', details: error.message });
  }
});

// Start server
app.listen(PORT, () => {
  console.log(`🚀 PROXY SERVER: Running on http://localhost:${PORT}`);
  console.log(`🔗 PROXY SERVER: Forwarding to backend at ${BACKEND_URL}`);
  console.log(`📋 PROXY SERVER: Health check available at http://localhost:${PORT}/health`);
});

export default app;