import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Separator } from "@/components/ui/separator";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Eye, ThumbsUp, ThumbsDown, Clock, Calendar } from "lucide-react";
import type { Recording, Feedback } from "@shared/schema";

interface RecipeWithFeedback extends Recording {
  feedback?: Feedback[];
}

export default function RecipesPage() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [selectedRecipe, setSelectedRecipe] = useState<RecipeWithFeedback | null>(null);
  const [feedbackText, setFeedbackText] = useState("");

  // Fetch all recipes
  const { data: recipes = [], isLoading } = useQuery<RecipeWithFeedback[]>({
    queryKey: ["/api/recipes"],
    retry: 1,
  });

  // Submit feedback mutation
  const submitFeedbackMutation = useMutation({
    mutationFn: async (feedback: { recordingId: string; activityUrl: string; inferredIntent: string; userCorrection?: string; wasCorrect: boolean }) => {
      await apiRequest("/api/feedback", "POST", feedback);
    },
    onSuccess: () => {
      toast({
        title: "Feedback submitted",
        description: "Thank you for helping improve our AI analysis!",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/recipes"] });
      setFeedbackText("");
    },
    onError: (error) => {
      toast({
        title: "Failed to submit feedback",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const formatDate = (dateString: string | null) => {
    if (!dateString) return "Unknown";
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
      hour: "2-digit",
      minute: "2-digit",
    });
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "completed": return "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200";
      case "processing": return "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200";
      case "failed": return "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200";
      default: return "bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200";
    }
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6">
        <div className="grid gap-4">
          {[...Array(3)].map((_, i) => (
            <Card key={i} className="animate-pulse">
              <CardHeader>
                <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                <div className="h-3 bg-gray-100 rounded w-1/2"></div>
              </CardHeader>
              <CardContent>
                <div className="h-20 bg-gray-100 rounded"></div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6">
      <div className="mb-8">
        <h1 className="text-3xl font-bold tracking-tight">Your Saved Recipes</h1>
        <p className="text-muted-foreground mt-2">
          View and manage your AI-generated step-by-step guides from recorded browser activities
        </p>
      </div>

      {recipes.length === 0 ? (
        <Card>
          <CardContent className="text-center py-12">
            <h3 className="text-lg font-medium mb-2">No recipes yet</h3>
            <p className="text-muted-foreground mb-4">
              Install the ActivityRecorder Chrome extension to start recording browser activities and generating recipes.
            </p>
            <Button onClick={() => window.open("chrome://extensions/", "_blank")}>
              Open Extensions Page
            </Button>
          </CardContent>
        </Card>
      ) : (
        <div className="grid gap-6">
          {recipes.map((recipe) => {
            const plan = recipe.generatedPlan as any;
            const capturedData = recipe.capturedData as any;
            
            return (
              <Card key={recipe.id} className="hover:shadow-md transition-shadow">
                <CardHeader className="pb-4">
                  <div className="flex justify-between items-start">
                    <div className="flex-1">
                      <CardTitle className="text-xl mb-2">{recipe.taskDescription}</CardTitle>
                      <CardDescription className="flex items-center gap-4 text-sm">
                        <span className="flex items-center gap-1">
                          <Calendar className="h-4 w-4" />
                          {formatDate(recipe.createdAt)}
                        </span>
                        {plan?.steps && (
                          <span className="flex items-center gap-1">
                            <Clock className="h-4 w-4" />
                            {plan.steps.length} steps
                          </span>
                        )}
                      </CardDescription>
                    </div>
                    <Badge className={getStatusColor(recipe.status)}>
                      {recipe.status}
                    </Badge>
                  </div>
                </CardHeader>

                <CardContent>
                  {plan?.summary && (
                    <p className="text-sm text-muted-foreground mb-4 line-clamp-2">
                      {plan.summary}
                    </p>
                  )}

                  <div className="flex justify-between items-center">
                    <div className="flex gap-2">
                      {capturedData?.activities?.length > 0 && (
                        <Badge variant="outline">
                          {capturedData.activities.length} activities captured
                        </Badge>
                      )}
                      {plan?.confidence && (
                        <Badge variant="outline">
                          {Math.round(plan.confidence * 100)}% confidence
                        </Badge>
                      )}
                    </div>

                    <Dialog>
                      <DialogTrigger asChild>
                        <Button 
                          variant="outline" 
                          size="sm"
                          onClick={() => setSelectedRecipe(recipe)}
                        >
                          <Eye className="h-4 w-4 mr-2" />
                          View Details
                        </Button>
                      </DialogTrigger>
                      <DialogContent className="max-w-4xl max-h-[90vh] overflow-hidden">
                        <DialogHeader>
                          <DialogTitle>{recipe.taskDescription}</DialogTitle>
                          <DialogDescription>
                            Generated on {formatDate(recipe.createdAt)} • {recipe.status}
                          </DialogDescription>
                        </DialogHeader>
                        
                        <ScrollArea className="h-[70vh] pr-4">
                          {selectedRecipe && <RecipeDetails recipe={selectedRecipe} onSubmitFeedback={submitFeedbackMutation} />}
                        </ScrollArea>
                      </DialogContent>
                    </Dialog>
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </div>
      )}
    </div>
  );
}

interface RecipeDetailsProps {
  recipe: RecipeWithFeedback;
  onSubmitFeedback: any;
}

function RecipeDetails({ recipe, onSubmitFeedback }: RecipeDetailsProps) {
  const [feedbackForm, setFeedbackForm] = useState<{
    activityUrl: string;
    inferredIntent: string;
    userCorrection: string;
    wasCorrect: boolean;
  } | null>(null);

  const plan = recipe.generatedPlan as any;
  const capturedData = recipe.capturedData as any;
  const screenshots = recipe.screenshots as any[] || [];

  const handleFeedbackSubmit = () => {
    if (feedbackForm) {
      onSubmitFeedback.mutate({
        recordingId: recipe.id,
        ...feedbackForm,
      });
      setFeedbackForm(null);
    }
  };

  return (
    <div className="space-y-6">
      {/* Summary */}
      {plan?.summary && (
        <div>
          <h3 className="font-semibold mb-2">Summary</h3>
          <p className="text-sm text-muted-foreground">{plan.summary}</p>
        </div>
      )}

      {/* Steps */}
      {plan?.steps && plan.steps.length > 0 && (
        <div>
          <h3 className="font-semibold mb-4">Steps</h3>
          <div className="space-y-4">
            {plan.steps.map((step: any, index: number) => (
              <div key={index} className="border rounded-lg p-4">
                <div className="flex items-start gap-3">
                  <div className="bg-primary text-primary-foreground rounded-full w-6 h-6 flex items-center justify-center text-sm font-medium">
                    {step.number || index + 1}
                  </div>
                  <div className="flex-1">
                    <h4 className="font-medium mb-1">{step.title}</h4>
                    <p className="text-sm text-muted-foreground mb-2">{step.description}</p>
                    {step.url && (
                      <Badge variant="outline" className="text-xs">
                        {step.url}
                      </Badge>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Screenshots */}
      {screenshots.length > 0 && (
        <div>
          <h3 className="font-semibold mb-4">Screenshots</h3>
          <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
            {screenshots.map((screenshot, index) => (
              <div key={index} className="border rounded-lg overflow-hidden">
                <img
                  src={`/api/screenshots/${screenshot.id || screenshot}`}
                  alt={`Screenshot ${index + 1}`}
                  className="w-full h-32 object-cover"
                  onError={(e) => {
                    const target = e.target as HTMLImageElement;
                    target.src = "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAwIiBoZWlnaHQ9IjEyMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMTAwJSIgaGVpZ2h0PSIxMDAlIiBmaWxsPSIjZjNmNGY2Ii8+PHRleHQgeD0iNTAlIiB5PSI1MCUiIGZvbnQtZmFtaWx5PSJBcmlhbCwgc2Fucy1zZXJpZiIgZm9udC1zaXplPSIxNCIgZmlsbD0iIzk5YTNhZiIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZHk9Ii4zZW0iPk5vIEltYWdlPC90ZXh0Pjwvc3ZnPg==";
                  }}
                />
              </div>
            ))}
          </div>
        </div>
      )}

      {/* AI Interpretations for Feedback */}
      {plan?.activityIntents && plan.activityIntents.length > 0 && (
        <div>
          <h3 className="font-semibold mb-4">AI Activity Analysis</h3>
          <div className="space-y-4">
            {plan.activityIntents.map((intent: any, index: number) => (
              <div key={index} className="border rounded-lg p-4 bg-muted/50">
                <div className="flex justify-between items-start mb-3">
                  <div>
                    <h4 className="font-medium text-sm">{intent.title}</h4>
                    <p className="text-xs text-muted-foreground">{intent.url}</p>
                  </div>
                  <Badge variant="outline" className="text-xs">
                    {Math.round((intent.confidence || 0) * 100)}% confident
                  </Badge>
                </div>
                
                <p className="text-sm mb-3">
                  <strong>AI interpreted:</strong> {intent.inferredIntent}
                </p>
                
                {intent.reasoning && (
                  <p className="text-xs text-muted-foreground mb-3">
                    <strong>Reasoning:</strong> {intent.reasoning}
                  </p>
                )}

                <div className="flex gap-2">
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={() => onSubmitFeedback.mutate({
                      recordingId: recipe.id,
                      activityUrl: intent.url,
                      inferredIntent: intent.inferredIntent,
                      wasCorrect: true
                    })}
                  >
                    <ThumbsUp className="h-3 w-3 mr-1" />
                    Correct
                  </Button>
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={() => setFeedbackForm({
                      activityUrl: intent.url,
                      inferredIntent: intent.inferredIntent,
                      userCorrection: "",
                      wasCorrect: false
                    })}
                  >
                    <ThumbsDown className="h-3 w-3 mr-1" />
                    Incorrect
                  </Button>
                </div>

                {feedbackForm && feedbackForm.activityUrl === intent.url && (
                  <div className="mt-4 space-y-3">
                    <Label htmlFor="correction">What was the actual intent?</Label>
                    <Textarea
                      id="correction"
                      placeholder="Describe what you were actually trying to do on this page..."
                      value={feedbackForm.userCorrection}
                      onChange={(e) => setFeedbackForm({ ...feedbackForm, userCorrection: e.target.value })}
                    />
                    <div className="flex gap-2">
                      <Button size="sm" onClick={handleFeedbackSubmit}>
                        Submit Feedback
                      </Button>
                      <Button size="sm" variant="ghost" onClick={() => setFeedbackForm(null)}>
                        Cancel
                      </Button>
                    </div>
                  </div>
                )}
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Raw Data */}
      {capturedData?.activities && (
        <div>
          <h3 className="font-semibold mb-4">Captured Activities ({capturedData.activities.length})</h3>
          <div className="text-xs bg-muted p-4 rounded-lg overflow-auto max-h-40">
            <pre>{JSON.stringify(capturedData.activities.slice(0, 5), null, 2)}</pre>
            {capturedData.activities.length > 5 && (
              <p className="mt-2 text-muted-foreground">... and {capturedData.activities.length - 5} more activities</p>
            )}
          </div>
        </div>
      )}
    </div>
  );
}