import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";

export default function DebugCapturePage() {
  const [taskDescription, setTaskDescription] = useState('');
  const [capturedActivities, setCapturedActivities] = useState([
    {
      timestamp: Date.now() - 300000,
      url: '',
      title: '',
      action: 'navigate',
      tabId: 1,
      screenshot: ''
    }
  ]);
  const [isProcessing, setIsProcessing] = useState(false);
  const [result, setResult] = useState<any>(null);

  // Task-agnostic preset examples for testing different workflows
  const presetExamples = {
    'research': {
      task: 'Researching laptop options for video editing work',
      activities: [
        {
          timestamp: Date.now() - 300000,
          url: 'https://www.apple.com/macbook-pro/',
          title: 'MacBook Pro - Apple',
          action: 'navigate',
          tabId: 1,
          screenshot: '' // Text-only analysis
        },
        {
          timestamp: Date.now() - 200000,
          url: 'https://www.youtube.com/watch?v=video-editing-review',
          title: 'Best Laptops for Video Editing 2024 - YouTube',
          action: 'navigate',
          tabId: 2,
          screenshot: '' // Text-only analysis
        },
        {
          timestamp: Date.now() - 100000,
          url: 'https://www.reddit.com/r/VideoEditing/comments/laptop-recommendations/',
          title: 'Laptop recommendations for video editing : VideoEditing',
          action: 'navigate',
          tabId: 3,
          screenshot: '' // Text-only analysis
        }
      ]
    },
    'shopping': {
      task: 'Finding a birthday gift for my partner who loves cooking',
      activities: [
        {
          timestamp: Date.now() - 400000,
          url: 'https://www.williams-sonoma.com/products/kitchen-tools/',
          title: 'Kitchen Tools & Gadgets - Williams Sonoma',
          action: 'navigate',
          tabId: 1,
          screenshot: '' // Text-only analysis
        },
        {
          timestamp: Date.now() - 300000,
          url: 'https://www.amazon.com/s?k=chef+knife+set',
          title: 'Amazon.com: chef knife set',
          action: 'navigate',
          tabId: 2,
          screenshot: '' // Text-only analysis
        }
      ]
    },
    'planning': {
      task: 'Planning a weekend trip to a nearby city',
      activities: [
        {
          timestamp: Date.now() - 500000,
          url: 'https://www.booking.com/city/us/san-francisco.html',
          title: 'San Francisco Hotels - Booking.com',
          action: 'navigate',
          tabId: 1,
          screenshot: '' // Text-only analysis
        },
        {
          timestamp: Date.now() - 400000,
          url: 'https://www.tripadvisor.com/Attractions-g60713-Activities-San_Francisco_California.html',
          title: 'Top Things to Do in San Francisco - TripAdvisor',
          action: 'navigate',
          tabId: 2,
          screenshot: '' // Text-only analysis
        },
        {
          timestamp: Date.now() - 300000,
          url: 'https://maps.google.com/search/restaurants+san+francisco',
          title: 'Restaurants in San Francisco - Google Maps',
          action: 'navigate',
          tabId: 3,
          screenshot: '' // Text-only analysis
        }
      ]
    }
  };

  const loadPresetExample = (exampleKey: string) => {
    const example = presetExamples[exampleKey as keyof typeof presetExamples];
    if (example) {
      setTaskDescription(example.task);
      setCapturedActivities(example.activities);
    }
  };

  const addActivity = () => {
    setCapturedActivities([...capturedActivities, {
      timestamp: Date.now(),
      url: '',
      title: '',
      action: 'navigate',
      tabId: capturedActivities.length + 1,
      screenshot: ''
    }]);
  };

  const updateActivity = (index: number, field: string, value: string | number) => {
    const updated = capturedActivities.map((activity, i) => 
      i === index ? { ...activity, [field]: value } : activity
    );
    setCapturedActivities(updated);
  };

  const removeActivity = (index: number) => {
    setCapturedActivities(capturedActivities.filter((_, i) => i !== index));
  };

  const generatePlan = async () => {
    if (!taskDescription.trim()) {
      alert('Please provide a task description');
      return;
    }

    if (capturedActivities.length === 0) {
      alert('Please add at least one activity');
      return;
    }

    setIsProcessing(true);
    setResult(null);

    try {
      console.log('=== Sending Debug Data ===');
      console.log('Task:', taskDescription);
      console.log('Activities:', capturedActivities);

      const response = await fetch('/api/recordings', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          taskDescription,
          capturedData: {
            sessionId: `debug_session_${Date.now()}`,
            startTime: Date.now() - 600000,
            endTime: Date.now(),
            activities: capturedActivities,
            tabs: capturedActivities.map(activity => ({
              id: activity.tabId,
              url: activity.url,
              title: activity.title
            }))
          }
        })
      });

      console.log('=== Server Response ===');
      const data = await response.json();
      console.log(data);

      if (!response.ok) {
        throw new Error(data.error || `HTTP ${response.status}`);
      }

      setResult(data.recording);
    } catch (error) {
      console.error('Error generating plan:', error);
      alert(`Failed to generate plan: ${error instanceof Error ? error.message : 'Unknown error'}`);
    } finally {
      setIsProcessing(false);
    }
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Debug Capture Interface</h1>
          <p className="text-gray-600 mt-2">Test AI plan generation with custom activities</p>
        </div>
        <div className="flex gap-2">
          <Button 
            onClick={() => loadPresetExample('research')}
            variant="outline"
            size="sm"
          >
            Load Research Example
          </Button>
          <Button 
            onClick={() => loadPresetExample('shopping')}
            variant="outline"
            size="sm"
          >
            Load Shopping Example
          </Button>
          <Button 
            onClick={() => loadPresetExample('planning')}
            variant="outline"
            size="sm"
          >
            Load Planning Example
          </Button>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Task Description</CardTitle>
          <CardDescription>What are you trying to accomplish?</CardDescription>
        </CardHeader>
        <CardContent>
          <Textarea
            placeholder="e.g., Finding the best laptop for video editing within my budget"
            value={taskDescription}
            onChange={(e) => setTaskDescription(e.target.value)}
            className="min-h-20"
          />
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            Captured Activities
            <Button onClick={addActivity} size="sm">Add Activity</Button>
          </CardTitle>
          <CardDescription>
            Simulate browser activities that would be captured by the extension
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          {capturedActivities.map((activity, index) => (
            <div key={index} className="p-4 border rounded-lg space-y-3">
              <div className="flex items-center justify-between">
                <Badge variant="outline">Activity {index + 1}</Badge>
                <Button 
                  onClick={() => removeActivity(index)} 
                  variant="destructive" 
                  size="sm"
                >
                  Remove
                </Button>
              </div>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>URL</Label>
                  <Input
                    placeholder="https://example.com"
                    value={activity.url}
                    onChange={(e) => updateActivity(index, 'url', e.target.value)}
                  />
                </div>
                
                <div className="space-y-2">
                  <Label>Page Title</Label>
                  <Input
                    placeholder="Example Page Title"
                    value={activity.title}
                    onChange={(e) => updateActivity(index, 'title', e.target.value)}
                  />
                </div>
                
                <div className="space-y-2">
                  <Label>Tab ID</Label>
                  <Input
                    type="number"
                    value={activity.tabId}
                    onChange={(e) => updateActivity(index, 'tabId', parseInt(e.target.value) || 1)}
                  />
                </div>
                
                <div className="space-y-2">
                  <Label>Screenshot (Base64 Data URL)</Label>
                  <Input
                    placeholder="data:image/jpeg;base64,..."
                    value={activity.screenshot}
                    onChange={(e) => updateActivity(index, 'screenshot', e.target.value)}
                  />
                </div>
              </div>
            </div>
          ))}
        </CardContent>
      </Card>

      <div className="flex justify-center">
        <Button 
          onClick={generatePlan}
          disabled={isProcessing}
          size="lg"
          className="px-8"
        >
          {isProcessing ? 'Processing...' : 'Generate Plan'}
        </Button>
      </div>

      {result && (
        <Card>
          <CardHeader>
            <CardTitle>Generated Plan</CardTitle>
            <CardDescription>AI-generated step-by-step guide</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div>
                <h3 className="font-semibold mb-2">Summary</h3>
                <p className="text-gray-700">{result.generatedPlan?.summary || 'No summary generated'}</p>
              </div>
              
              {result.generatedPlan?.steps && result.generatedPlan.steps.length > 0 && (
                <div>
                  <h3 className="font-semibold mb-2">Steps</h3>
                  <div className="space-y-2">
                    {result.generatedPlan.steps.map((step: any, index: number) => (
                      <div key={index} className="p-3 bg-gray-50 rounded">
                        <h4 className="font-medium">{step.number}. {step.title}</h4>
                        <p className="text-sm text-gray-600 mt-1">{step.description}</p>
                        {step.url && (
                          <a href={step.url} target="_blank" rel="noopener noreferrer" 
                             className="text-blue-600 text-sm hover:underline">
                            {step.url}
                          </a>
                        )}
                      </div>
                    ))}
                  </div>
                </div>
              )}
              
              {result.generatedPlan?.activityIntents && (
                <div>
                  <h3 className="font-semibold mb-2">Activity Analysis</h3>
                  <div className="space-y-2">
                    {result.generatedPlan.activityIntents.map((intent: any, index: number) => (
                      <div key={index} className="p-3 border-l-4 border-blue-500 bg-blue-50 rounded">
                        <div className="flex items-center justify-between mb-1">
                          <span className="font-medium text-sm">{intent.url}</span>
                          <div className="flex gap-2">
                            <Badge variant="outline" className="text-xs">
                              Relevance: {Math.round(intent.relevance * 100)}%
                            </Badge>
                            <Badge variant="outline" className="text-xs">
                              Confidence: {Math.round(intent.confidence * 100)}%
                            </Badge>
                          </div>
                        </div>
                        <p className="text-sm text-gray-700 mb-1">
                          <strong>Intent:</strong> {intent.inferredIntent}
                        </p>
                        <p className="text-xs text-gray-600">
                          <strong>Reasoning:</strong> {intent.reasoning}
                        </p>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}