import { useState, useEffect } from "react";
import { Link } from "wouter";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Progress } from "@/components/ui/progress";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { useMutation } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { 
  Play, 
  Square, 
  Bot, 
  Settings, 
  HelpCircle, 
  CheckCircle,
  Download,
  Bookmark,
  RotateCcw,
  Activity,
  ExternalLink,
  Clock,
  ChefHat,
  BookOpen
} from "lucide-react";

type RecordingState = 'idle' | 'recording' | 'taskDescription' | 'processing' | 'planDisplay';

interface GeneratedPlan {
  summary: string;
  steps: Array<{
    number: number;
    title: string;
    description: string;
    url?: string;
    duration?: string;
  }>;
  totalDuration?: string;
}

export default function Home() {
  const [recordingState, setRecordingState] = useState<RecordingState>('idle');
  const [taskDescription, setTaskDescription] = useState('');
  const [recordingStats, setRecordingStats] = useState({ duration: '00:00', actionCount: 0 });
  const [generatedPlan, setGeneratedPlan] = useState<GeneratedPlan | null>(null);
  const [activeTabs, setActiveTabs] = useState([
    { id: 1, title: 'Google Docs - Product Roadmap', url: 'https://docs.google.com/document/d/...' },
    { id: 2, title: 'Figma - Design System Components', url: 'https://figma.com/file/abc123...' }
  ]);
  
  const { toast } = useToast();

  // Handle Electron menu actions
  useEffect(() => {
    if (typeof window !== 'undefined' && window.electronAPI) {
      const handleMenuAction = (action: string) => {
        switch (action) {
          case 'new-recording':
            if (recordingState === 'idle') {
              handleStartRecording();
            }
            break;
          case 'open-recordings':
            // Navigate to recordings page
            window.location.href = '/recordings';
            break;
          case 'settings':
            // Open settings modal or navigate to settings
            toast({
              title: "Settings",
              description: "Settings panel coming soon!",
            });
            break;
        }
      };

      window.electronAPI.onMenuAction(handleMenuAction);
      
      return () => {
        window.electronAPI.removeAllListeners('menu-action');
      };
    }
  }, [recordingState]);

  const generatePlanMutation = useMutation({
    mutationFn: async (data: { taskDescription: string; capturedData: any }) => {
      const response = await apiRequest('POST', '/api/recordings', data);
      return response.json();
    },
    onSuccess: (data) => {
      if (data.success) {
        setGeneratedPlan(data.plan);
        setRecordingState('planDisplay');
        toast({
          title: "Plan Generated Successfully",
          description: "Your activity recipe is ready!",
        });
      } else {
        throw new Error(data.error || 'Failed to generate plan');
      }
    },
    onError: (error) => {
      setRecordingState('taskDescription');
      toast({
        variant: "destructive",
        title: "Generation Failed",
        description: (error as Error).message,
      });
    },
  });

  const handleStartRecording = () => {
    setRecordingState('recording');
    setRecordingStats({ duration: '00:00', actionCount: 0 });
    
    // Simulate realistic browser tabs being opened during recording
    const commonWorkflowTabs = [
      { id: 1, title: 'Google Slides - Project Presentation', url: 'https://docs.google.com/presentation/d/abc123/edit' },
      { id: 2, title: 'Unsplash - Stock Photos', url: 'https://unsplash.com/photos/business-meeting' },
      { id: 3, title: 'Slack - Team Communication', url: 'https://app.slack.com/client/workspace/general' },
      { id: 4, title: 'Google Calendar - Schedule Meeting', url: 'https://calendar.google.com/calendar/render' },
      { id: 5, title: 'Canva - Design Elements', url: 'https://www.canva.com/design/templates' }
    ];
    
    // Start with initial tabs
    setActiveTabs(commonWorkflowTabs.slice(0, 2));
    let tabIndex = 2;
    
    // Simulate recording timer and tab changes
    let seconds = 0;
    let actions = 0;
    const timer = setInterval(() => {
      seconds++;
      actions += Math.random() > 0.7 ? 1 : 0;
      const minutes = Math.floor(seconds / 60);
      const remainingSeconds = seconds % 60;
      setRecordingStats({
        duration: `${minutes.toString().padStart(2, '0')}:${remainingSeconds.toString().padStart(2, '0')}`,
        actionCount: actions
      });
      
      // Simulate new tabs being opened during recording
      if (seconds % 4 === 0 && tabIndex < commonWorkflowTabs.length) {
        setActiveTabs(commonWorkflowTabs.slice(0, tabIndex + 1));
        tabIndex++;
      }
    }, 1000);

    // Store timer ID for cleanup
    (window as any).recordingTimer = timer;
  };

  const handleStopRecording = () => {
    if ((window as any).recordingTimer) {
      clearInterval((window as any).recordingTimer);
    }
    setRecordingState('taskDescription');
  };

  const handleGeneratePlan = () => {
    if (!taskDescription.trim()) {
      toast({
        variant: "destructive",
        title: "Description Required",
        description: "Please describe what you accomplished during the recording.",
      });
      return;
    }

    setRecordingState('processing');
    
    // Mock captured data - in real implementation, this would come from the Chrome extension
    const capturedData = {
      sessionId: 'session_' + Date.now(),
      startTime: Date.now() - 600000, // 10 minutes ago
      endTime: Date.now(),
      activities: [
        {
          timestamp: Date.now() - 600000,
          url: 'https://docs.google.com/presentation/d/abc123/edit',
          title: 'Google Slides - Project Presentation',
          action: 'create',
          tabId: 1
        },
        {
          timestamp: Date.now() - 500000,
          url: 'https://unsplash.com/photos/business-meeting',
          title: 'Unsplash - Stock Photos',
          action: 'navigate',
          tabId: 2
        },
        {
          timestamp: Date.now() - 400000,
          url: 'https://app.slack.com/client/workspace/general',
          title: 'Slack - Team Communication',
          action: 'navigate',
          tabId: 3
        },
        {
          timestamp: Date.now() - 300000,
          url: 'https://calendar.google.com/calendar/render',
          title: 'Google Calendar - Schedule Meeting',
          action: 'navigate',
          tabId: 4
        },
        {
          timestamp: Date.now() - 200000,
          url: 'https://www.canva.com/design/templates',
          title: 'Canva - Design Elements',
          action: 'navigate',
          tabId: 5
        }
      ],
      tabs: [
        { id: 1, url: 'https://docs.google.com/presentation/d/abc123/edit', title: 'Google Slides - Project Presentation' },
        { id: 2, url: 'https://unsplash.com/photos/business-meeting', title: 'Unsplash - Stock Photos' },
        { id: 3, url: 'https://app.slack.com/client/workspace/general', title: 'Slack - Team Communication' },
        { id: 4, url: 'https://calendar.google.com/calendar/render', title: 'Google Calendar - Schedule Meeting' },
        { id: 5, url: 'https://www.canva.com/design/templates', title: 'Canva - Design Elements' }
      ]
    };

    generatePlanMutation.mutate({
      taskDescription,
      capturedData
    });
  };

  const handleDiscard = () => {
    setRecordingState('idle');
    setTaskDescription('');
    setGeneratedPlan(null);
  };

  const handleNewRecording = () => {
    setRecordingState('idle');
    setTaskDescription('');
    setGeneratedPlan(null);
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Navigation Header */}
      <header className="bg-white border-b">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <Link href="/">
                <div className="flex items-center space-x-2 cursor-pointer">
                  <Bot className="w-6 h-6 text-blue-600" />
                  <span className="font-bold text-lg text-gray-900">ActivityRecorder</span>
                </div>
              </Link>
            </div>
            <nav className="flex items-center space-x-4">
              <Link href="/recipes">
                <Button variant="ghost" size="sm" className="flex items-center gap-1">
                  <ChefHat className="h-4 w-4" />
                  Recipes
                </Button>
              </Link>
              <Link href="/recordings">
                <Button variant="ghost" size="sm" className="flex items-center gap-1">
                  <BookOpen className="h-4 w-4" />
                  Recordings
                </Button>
              </Link>
              <div className="flex items-center space-x-1 text-sm text-gray-600">
                <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                <span>Extension Connected</span>
              </div>
            </nav>
          </div>
        </div>
      </header>

      <div className="container mx-auto px-4 py-8">
        <div className="max-w-md mx-auto bg-white shadow-lg rounded-lg overflow-hidden">
          {/* Content */}
          <div className="p-6">
          {/* Idle State */}
          {recordingState === 'idle' && (
            <div className="text-center animate-fade-in">
              <div className="mb-4">
                <Button
                  onClick={handleStartRecording}
                  className="w-16 h-16 rounded-full bg-md-error hover:bg-red-600 text-white shadow-lg"
                  size="icon"
                >
                  <Play className="w-6 h-6" />
                </Button>
              </div>
              <h2 className="text-lg font-medium text-md-secondary mb-2">Ready to Record</h2>
              <p className="text-sm text-gray-600 mb-4">
                Click the record button to start capturing your browser activities across all tabs.
              </p>
              
              <div className="flex gap-2 justify-center mb-4">
                <Link href="/recipes">
                  <Button variant="outline" size="sm" className="flex items-center gap-1">
                    <ChefHat className="h-3 w-3" />
                    Recipes
                  </Button>
                </Link>
                <Link href="/recordings">
                  <Button variant="outline" size="sm" className="flex items-center gap-1">
                    <BookOpen className="h-3 w-3" />
                    All Records
                  </Button>
                </Link>
              </div>
              
              <div className="bg-gray-50 rounded-lg p-3 mb-4">
                <div className="flex items-center justify-center space-x-4 text-xs text-gray-600">
                  <div className="flex items-center space-x-1">
                    <Activity className="w-4 h-4" />
                    <span>Multi-tab tracking</span>
                  </div>
                  <div className="flex items-center space-x-1">
                    <Bot className="w-4 h-4" />
                    <span>AI-powered analysis</span>
                  </div>
                </div>
              </div>

              <div className="flex flex-col space-y-2">
                <Button 
                  onClick={() => window.location.href = '/recordings'} 
                  variant="outline" 
                  className="w-full flex items-center space-x-2"
                >
                  <Bookmark className="w-4 h-4" />
                  <span>View Saved Recordings</span>
                </Button>
                <Button 
                  onClick={() => window.location.href = '/debug'} 
                  variant="outline" 
                  className="w-full flex items-center space-x-2"
                >
                  <Settings className="w-4 h-4" />
                  <span>Debug Activity Capture</span>
                </Button>
              </div>
            </div>
          )}

          {/* Recording State */}
          {recordingState === 'recording' && (
            <div className="text-center animate-fade-in">
              <div className="mb-4">
                <Button
                  onClick={handleStopRecording}
                  className="w-16 h-16 rounded-full bg-md-error hover:bg-red-600 text-white shadow-lg recording-indicator"
                  size="icon"
                >
                  <Square className="w-6 h-6" />
                </Button>
              </div>
              <h2 className="text-lg font-medium text-md-error mb-2">Recording Active</h2>
              
              <div className="bg-red-50 rounded-lg p-3 mb-4">
                <div className="grid grid-cols-2 gap-4 text-center">
                  <div>
                    <div className="text-xl font-bold text-md-error">{recordingStats.duration}</div>
                    <div className="text-xs text-gray-600">Duration</div>
                  </div>
                  <div>
                    <div className="text-xl font-bold text-md-error">{recordingStats.actionCount}</div>
                    <div className="text-xs text-gray-600">Actions</div>
                  </div>
                </div>
              </div>

              <div className="space-y-2">
                <h3 className="text-sm font-medium text-gray-700">Active Tabs</h3>
                {activeTabs.map((tab) => (
                  <div key={tab.id} className="bg-gray-50 rounded px-3 py-2 text-left">
                    <div className="flex items-center space-x-2">
                      <div className="w-2 h-2 bg-md-success rounded-full"></div>
                      <span className="text-sm text-gray-900 truncate">{tab.title}</span>
                    </div>
                    <div className="text-xs text-gray-500 truncate ml-4">{tab.url}</div>
                  </div>
                ))}
              </div>
            </div>
          )}

          {/* Task Description State */}
          {recordingState === 'taskDescription' && (
            <div className="animate-fade-in">
              <h2 className="text-lg font-medium text-md-secondary mb-3">Describe Your Task</h2>
              <p className="text-sm text-gray-600 mb-4">
                What did you accomplish during this recording session?
              </p>
              
              <div className="space-y-4">
                <div>
                  <Label htmlFor="taskDescription" className="text-sm font-medium text-gray-700">
                    Task Description
                  </Label>
                  <Textarea
                    id="taskDescription"
                    value={taskDescription}
                    onChange={(e) => setTaskDescription(e.target.value)}
                    placeholder="Example: I created a new product roadmap document in Google Docs, added quarterly milestones, and shared it with the design team through Figma..."
                    className="mt-2 h-24 resize-none"
                  />
                </div>
                
                <div className="flex space-x-2">
                  <Button
                    onClick={handleGeneratePlan}
                    disabled={!taskDescription.trim()}
                    className="flex-1 bg-md-primary hover:bg-blue-700"
                  >
                    Generate Plan
                  </Button>
                  <Button
                    onClick={handleDiscard}
                    variant="outline"
                  >
                    Discard
                  </Button>
                </div>
              </div>
            </div>
          )}

          {/* Processing State */}
          {recordingState === 'processing' && (
            <div className="text-center animate-fade-in">
              <div className="mb-4">
                <div className="mx-auto w-12 h-12 border-4 border-md-primary border-t-transparent rounded-full animate-spin"></div>
              </div>
              <h2 className="text-lg font-medium text-md-secondary mb-2">Generating Your Plan</h2>
              <p className="text-sm text-gray-600 mb-4">
                Our AI is analyzing your activities and creating a step-by-step recipe...
              </p>
              
              <div className="bg-blue-50 rounded-lg p-3">
                <div className="text-sm text-gray-700">
                  <div className="flex items-center justify-center space-x-2 mb-2">
                    <Bot className="w-4 h-4 text-md-primary" />
                    <span>Processing browser activities</span>
                  </div>
                  <Progress value={65} className="w-full h-2" />
                </div>
              </div>
            </div>
          )}

          {/* Plan Display State */}
          {recordingState === 'planDisplay' && generatedPlan && (
            <div className="animate-fade-in">
              <div className="flex items-center justify-between mb-4">
                <h2 className="text-lg font-medium text-md-secondary">Your Activity Recipe</h2>
                <Button
                  onClick={handleNewRecording}
                  variant="ghost"
                  size="icon"
                  className="text-md-primary hover:text-blue-700"
                >
                  <RotateCcw className="w-4 h-4" />
                </Button>
              </div>
              
              <div className="space-y-3">
                <div className="bg-md-success text-white rounded-lg p-3 text-sm">
                  <div className="flex items-center space-x-2">
                    <CheckCircle className="w-4 h-4" />
                    <span className="font-medium">Plan Generated Successfully</span>
                  </div>
                </div>

                <div className="bg-gray-50 rounded-lg p-3">
                  <h3 className="font-medium text-sm text-gray-700 mb-2">Task Summary</h3>
                  <p className="text-sm text-gray-600">{generatedPlan.summary}</p>
                </div>

                <div className="space-y-2">
                  <h3 className="font-medium text-sm text-gray-700">Step-by-Step Recipe</h3>
                  
                  {generatedPlan.steps.map((step) => (
                    <Card key={step.number} className="border border-gray-200">
                      <CardContent className="p-3">
                        <div className="flex items-start space-x-3">
                          <Badge className="bg-md-primary text-white rounded-full w-6 h-6 flex items-center justify-center text-xs font-medium">
                            {step.number}
                          </Badge>
                          <div className="flex-1">
                            <h4 className="font-medium text-sm text-gray-900 mb-1">{step.title}</h4>
                            <p className="text-xs text-gray-600 mb-2">{step.description}</p>
                            <div className="flex items-center space-x-2 text-xs text-gray-500">
                              {step.url && (
                                <>
                                  <ExternalLink className="w-3 h-3" />
                                  <span>{new URL(step.url).hostname}</span>
                                  <span>•</span>
                                </>
                              )}
                              {step.duration && (
                                <>
                                  <Clock className="w-3 h-3" />
                                  <span>{step.duration}</span>
                                </>
                              )}
                            </div>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>

                <div className="flex space-x-2 pt-2">
                  <Button className="flex-1 bg-md-success hover:bg-green-600 text-white">
                    <Bookmark className="w-4 h-4 mr-1" />
                    Save Recipe
                  </Button>
                  <Button variant="outline" size="icon">
                    <Download className="w-4 h-4" />
                  </Button>
                </div>
              </div>
            </div>
          )}
          </div>

          {/* Footer */}
          <div className="border-t border-gray-100 px-4 py-3 bg-gray-50">
            <div className="flex items-center justify-between text-xs text-gray-600">
              <div className="flex items-center space-x-3">
                <Button variant="ghost" size="icon" className="w-8 h-8">
                  <Settings className="w-4 h-4" />
                </Button>
                <Button variant="ghost" size="icon" className="w-8 h-8">
                  <HelpCircle className="w-4 h-4" />
                </Button>
              </div>
              <div className="text-right">
                <div>v1.0.0</div>
                <Link href="/test">
                  <span className="text-xs text-blue-600 hover:text-blue-800 cursor-pointer">Test Mode</span>
                </Link>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
