import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { 
  Download, 
  Chrome, 
  Bot, 
  Activity, 
  Eye, 
  Zap, 
  Shield, 
  ChefHat,
  PlayCircle,
  CheckCircle,
  ArrowRight,
  Github,
  ExternalLink
} from "lucide-react";
import { Link } from "wouter";

export default function Landing() {
  const handleDownloadExtension = () => {
    // Download the pre-packaged extension
    window.open('/activityrecorder-extension.zip', '_blank');
  };

  const features = [
    {
      icon: <Activity className="h-6 w-6 text-blue-600" />,
      title: "Multi-Tab Activity Capture",
      description: "Automatically tracks your browser activities across all tabs with intelligent interaction detection"
    },
    {
      icon: <Eye className="h-6 w-6 text-purple-600" />,
      title: "Visual Context Analysis",
      description: "Uses screenshot capture and GPT-4o vision to understand what you're viewing and working with"
    },
    {
      icon: <Bot className="h-6 w-6 text-green-600" />,
      title: "AI-Powered Recipe Generation",
      description: "Converts your browser sessions into step-by-step guides that others can follow"
    },
    {
      icon: <Shield className="h-6 w-6 text-orange-600" />,
      title: "Privacy-First Design",
      description: "All processing happens securely with no permanent storage of sensitive browsing data"
    },
    {
      icon: <Zap className="h-6 w-6 text-yellow-600" />,
      title: "Real-Time Processing",
      description: "Instant activity streaming and analysis for immediate feedback and recipe generation"
    },
    {
      icon: <ChefHat className="h-6 w-6 text-red-600" />,
      title: "Operator Intelligence",
      description: "Understands intent and context to create actionable workflows, not just interaction logs"
    }
  ];

  const useCases = [
    {
      title: "Onboarding Documentation",
      description: "Create step-by-step guides for new team members by recording your workflow once",
      example: "Set up development environment, configure tools, first deployment"
    },
    {
      title: "Process Standardization", 
      description: "Capture and standardize repetitive business processes across your organization",
      example: "Customer onboarding flow, data analysis routine, report generation"
    },
    {
      title: "Training Content Creation",
      description: "Generate training materials by simply performing the task you want to teach",
      example: "Software tutorials, admin procedures, troubleshooting guides"
    },
    {
      title: "Knowledge Transfer",
      description: "Preserve institutional knowledge when team members transition roles",
      example: "Complex workflows, tool configurations, vendor processes"
    }
  ];

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-purple-50">
      {/* Header */}
      <header className="border-b bg-white/80 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <div className="p-2 bg-blue-600 rounded-lg">
                <Bot className="h-6 w-6 text-white" />
              </div>
              <div>
                <h1 className="text-xl font-bold text-gray-900">ActivityRecorder</h1>
                <p className="text-sm text-gray-600">AI-Powered Browser Workflow Documentation</p>
              </div>
            </div>
            <div className="flex items-center space-x-4">
              <Link href="/recipes">
                <Button variant="ghost" className="flex items-center gap-2">
                  <ChefHat className="h-4 w-4" />
                  Browse Recipes
                </Button>
              </Link>
              <Button onClick={handleDownloadExtension} className="flex items-center gap-2">
                <Download className="h-4 w-4" />
                Download Extension
              </Button>
            </div>
          </div>
        </div>
      </header>

      {/* Hero Section */}
      <section className="py-20 px-4">
        <div className="container mx-auto text-center">
          <Badge variant="secondary" className="mb-4">
            <PlayCircle className="h-3 w-3 mr-1" />
            AI-Powered Workflow Intelligence
          </Badge>
          <h1 className="text-5xl font-bold text-gray-900 mb-6">
            Turn Your Browser Activities Into
            <span className="text-blue-600"> Actionable Recipes</span>
          </h1>
          <p className="text-xl text-gray-600 mb-8 max-w-3xl mx-auto">
            ActivityRecorder captures your browser workflows across multiple tabs, uses AI vision to understand context, 
            and generates step-by-step "operator recipes" that others can follow to achieve the same results.
          </p>
          
          <div className="flex flex-col sm:flex-row gap-4 justify-center items-center mb-12">
            <Button 
              onClick={handleDownloadExtension}
              size="lg" 
              className="flex items-center gap-3 text-lg px-8 py-6"
            >
              <Chrome className="h-5 w-5" />
              Download Chrome Extension
              <ArrowRight className="h-4 w-4" />
            </Button>
            <Link href="/recipes">
              <Button variant="outline" size="lg" className="flex items-center gap-2 text-lg px-8 py-6">
                <ChefHat className="h-5 w-5" />
                View Sample Recipes
              </Button>
            </Link>
          </div>

          {/* Demo Video Placeholder */}
          <div className="max-w-4xl mx-auto">
            <div className="relative bg-gray-900 rounded-xl overflow-hidden shadow-2xl">
              <div className="aspect-video flex items-center justify-center">
                <div className="text-center text-white">
                  <PlayCircle className="h-16 w-16 mx-auto mb-4 opacity-70" />
                  <p className="text-lg">Watch Demo Video</p>
                  <p className="text-sm opacity-70">See ActivityRecorder in action</p>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* How It Works */}
      <section className="py-16 px-4 bg-white">
        <div className="container mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">How It Works</h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Three simple steps to transform your browser activities into reusable workflows
            </p>
          </div>
          
          <div className="grid md:grid-cols-3 gap-8">
            <div className="text-center">
              <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Activity className="h-8 w-8 text-blue-600" />
              </div>
              <h3 className="text-xl font-semibold mb-2">1. Record Your Workflow</h3>
              <p className="text-gray-600">
                Install the extension and click record. Work naturally across multiple browser tabs 
                while ActivityRecorder captures your actions and screenshots.
              </p>
            </div>
            
            <div className="text-center">
              <div className="w-16 h-16 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Bot className="h-8 w-8 text-purple-600" />
              </div>
              <h3 className="text-xl font-semibold mb-2">2. AI Analyzes Context</h3>
              <p className="text-gray-600">
                Our AI uses GPT-4o vision to understand what you're doing, not just where you're clicking. 
                It analyzes visual content and interaction patterns.
              </p>
            </div>
            
            <div className="text-center">
              <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <ChefHat className="h-8 w-8 text-green-600" />
              </div>
              <h3 className="text-xl font-semibold mb-2">3. Generate Operator Recipe</h3>
              <p className="text-gray-600">
                Get a structured, step-by-step guide that others can follow to achieve the same results, 
                complete with context and actionable instructions.
              </p>
            </div>
          </div>
        </div>
      </section>

      {/* Features Grid */}
      <section className="py-16 px-4 bg-gray-50">
        <div className="container mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Powerful Features</h2>
            <p className="text-lg text-gray-600">
              Built for teams who need to scale knowledge and standardize processes
            </p>
          </div>
          
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {features.map((feature, index) => (
              <Card key={index} className="border-0 shadow-md hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className="flex items-center gap-3">
                    {feature.icon}
                    <CardTitle className="text-lg">{feature.title}</CardTitle>
                  </div>
                </CardHeader>
                <CardContent>
                  <CardDescription className="text-gray-600">
                    {feature.description}
                  </CardDescription>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Use Cases */}
      <section className="py-16 px-4 bg-white">
        <div className="container mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Use Cases</h2>
            <p className="text-lg text-gray-600">
              Perfect for teams who need to document, standardize, and share complex workflows
            </p>
          </div>
          
          <div className="grid md:grid-cols-2 gap-8">
            {useCases.map((useCase, index) => (
              <Card key={index} className="border-l-4 border-l-blue-600">
                <CardHeader>
                  <CardTitle className="text-xl">{useCase.title}</CardTitle>
                  <CardDescription className="text-gray-600">
                    {useCase.description}
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="bg-blue-50 rounded-lg p-3">
                    <p className="text-sm text-blue-800 font-medium">Example:</p>
                    <p className="text-sm text-blue-700">{useCase.example}</p>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Installation Guide */}
      <section className="py-16 px-4 bg-blue-600 text-white">
        <div className="container mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold mb-4">Quick Installation</h2>
            <p className="text-lg text-blue-100">
              Get started with ActivityRecorder in under 2 minutes
            </p>
          </div>
          
          <div className="max-w-3xl mx-auto">
            <div className="grid md:grid-cols-2 gap-8">
              <div className="space-y-4">
                <h3 className="text-xl font-semibold flex items-center gap-2">
                  <Chrome className="h-5 w-5" />
                  Chrome Extension
                </h3>
                <div className="space-y-3">
                  <div className="flex items-start gap-3">
                    <div className="w-6 h-6 bg-white text-blue-600 rounded-full flex items-center justify-center text-sm font-bold mt-0.5">1</div>
                    <p>Download the extension package from the button above</p>
                  </div>
                  <div className="flex items-start gap-3">
                    <div className="w-6 h-6 bg-white text-blue-600 rounded-full flex items-center justify-center text-sm font-bold mt-0.5">2</div>
                    <p>Open Chrome and go to chrome://extensions/</p>
                  </div>
                  <div className="flex items-start gap-3">
                    <div className="w-6 h-6 bg-white text-blue-600 rounded-full flex items-center justify-center text-sm font-bold mt-0.5">3</div>
                    <p>Enable "Developer mode" and click "Load unpacked"</p>
                  </div>
                  <div className="flex items-start gap-3">
                    <div className="w-6 h-6 bg-white text-blue-600 rounded-full flex items-center justify-center text-sm font-bold mt-0.5">4</div>
                    <p>Select the extracted extension folder</p>
                  </div>
                </div>
              </div>
              
              <div className="space-y-4">
                <h3 className="text-xl font-semibold flex items-center gap-2">
                  <Bot className="h-5 w-5" />
                  Web Portal Access
                </h3>
                <div className="space-y-3">
                  <div className="flex items-start gap-3">
                    <CheckCircle className="h-5 w-5 mt-1" />
                    <p>Access recipes and recordings through this web portal</p>
                  </div>
                  <div className="flex items-start gap-3">
                    <CheckCircle className="h-5 w-5 mt-1" />
                    <p>View generated recipes from any device</p>
                  </div>
                  <div className="flex items-start gap-3">
                    <CheckCircle className="h-5 w-5 mt-1" />
                    <p>Share workflow guides with your team</p>
                  </div>
                  <div className="flex items-start gap-3">
                    <CheckCircle className="h-5 w-5 mt-1" />
                    <p>Provide feedback to improve AI analysis</p>
                  </div>
                </div>
              </div>
            </div>
            
            <div className="text-center mt-8">
              <Button 
                onClick={handleDownloadExtension}
                size="lg" 
                variant="secondary"
                className="text-blue-600"
              >
                <Download className="h-5 w-5 mr-2" />
                Download Chrome Extension Now
              </Button>
            </div>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-12 px-4">
        <div className="container mx-auto">
          <div className="grid md:grid-cols-4 gap-8">
            <div className="md:col-span-2">
              <div className="flex items-center space-x-3 mb-4">
                <div className="p-2 bg-blue-600 rounded-lg">
                  <Bot className="h-6 w-6 text-white" />
                </div>
                <div>
                  <h3 className="text-xl font-bold">ActivityRecorder</h3>
                  <p className="text-gray-400">AI-Powered Workflow Documentation</p>
                </div>
              </div>
              <p className="text-gray-400 mb-4">
                Transform your browser activities into actionable operator recipes that teams can follow. 
                Powered by AI vision and intelligent workflow analysis.
              </p>
            </div>
            
            <div>
              <h4 className="font-semibold mb-4">Product</h4>
              <div className="space-y-2">
                <Link href="/recipes" className="block text-gray-400 hover:text-white transition-colors">
                  Browse Recipes
                </Link>
                <Link href="/recordings" className="block text-gray-400 hover:text-white transition-colors">
                  All Recordings
                </Link>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Documentation
                </a>
              </div>
            </div>
            
            <div>
              <h4 className="font-semibold mb-4">Support</h4>
              <div className="space-y-2">
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Help Center
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors">
                  Contact Us
                </a>
                <a href="#" className="block text-gray-400 hover:text-white transition-colors flex items-center gap-1">
                  GitHub <ExternalLink className="h-3 w-3" />
                </a>
              </div>
            </div>
          </div>
          
          <div className="border-t border-gray-800 mt-8 pt-8 text-center text-gray-400">
            <p>&copy; 2025 ActivityRecorder. All rights reserved.</p>
          </div>
        </div>
      </footer>
    </div>
  );
}