import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Link } from "wouter";
import { ExternalLink, MessageSquare, Clock, CheckCircle, XCircle } from "lucide-react";

interface Recording {
  id: string;
  taskDescription: string;
  status: "processing" | "completed" | "failed";
  createdAt: string;
  completedAt?: string;
  generatedPlan?: {
    summary: string;
    steps: Array<{
      number: number;
      title: string;
      description: string;
      duration: string;
      completed: boolean;
    }>;
    confidence: number;
  };
}

export default function RecordingsPage() {
  const { data: recordingsData, isLoading } = useQuery<{ success: boolean; recordings: Recording[] }>({
    queryKey: ["/api/recordings"],
  });

  const recordings = recordingsData?.recordings || [];

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "completed":
        return <CheckCircle className="h-4 w-4 text-green-500" />;
      case "failed":
        return <XCircle className="h-4 w-4 text-red-500" />;
      default:
        return <Clock className="h-4 w-4 text-yellow-500" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "completed":
        return "bg-green-100 text-green-800";
      case "failed":
        return "bg-red-100 text-red-800";
      default:
        return "bg-yellow-100 text-yellow-800";
    }
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6">
        <div className="mb-6">
          <h1 className="text-3xl font-bold">Saved Activity Recordings</h1>
          <p className="text-gray-600 mt-2">Loading your recorded browser activities...</p>
        </div>
        <div className="space-y-4">
          {[1, 2, 3].map((i) => (
            <Card key={i} className="animate-pulse">
              <CardHeader>
                <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                <div className="h-3 bg-gray-200 rounded w-1/2"></div>
              </CardHeader>
              <CardContent>
                <div className="h-3 bg-gray-200 rounded w-full mb-2"></div>
                <div className="h-3 bg-gray-200 rounded w-2/3"></div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6">
      <div className="mb-6">
        <h1 className="text-3xl font-bold">Saved Activity Recordings</h1>
        <p className="text-gray-600 mt-2">
          View your recorded browser activities and the AI-generated step-by-step plans
        </p>
      </div>

      <div className="mb-4 flex gap-2">
        <Link href="/">
          <Button variant="outline">← Back to Home</Button>
        </Link>
        <Link href="/test">
          <Button variant="outline">Test Recording</Button>
        </Link>
      </div>

      {recordings.length === 0 ? (
        <Card>
          <CardContent className="pt-6">
            <div className="text-center py-8">
              <h3 className="text-lg font-medium mb-2">No recordings yet</h3>
              <p className="text-gray-600 mb-4">
                Start by testing the holistic planning system or installing the Chrome extension
              </p>
              <Link href="/test">
                <Button>Create Test Recording</Button>
              </Link>
            </div>
          </CardContent>
        </Card>
      ) : (
        <div className="space-y-4">
          {recordings.map((recording) => (
            <Card key={recording.id}>
              <CardHeader>
                <div className="flex items-start justify-between">
                  <div className="flex-1">
                    <CardTitle className="text-lg mb-2">{recording.taskDescription}</CardTitle>
                    <div className="flex items-center gap-2 text-sm text-gray-600">
                      {getStatusIcon(recording.status)}
                      <Badge className={getStatusColor(recording.status)}>
                        {recording.status}
                      </Badge>
                      <span>•</span>
                      <span>Created {new Date(recording.createdAt).toLocaleDateString()}</span>
                      {recording.completedAt && (
                        <>
                          <span>•</span>
                          <span>Completed {new Date(recording.completedAt).toLocaleDateString()}</span>
                        </>
                      )}
                    </div>
                  </div>
                </div>
              </CardHeader>

              <CardContent>
                {recording.generatedPlan ? (
                  <div className="space-y-4">
                    <div>
                      <h4 className="font-medium mb-2">AI-Generated Plan Summary</h4>
                      <p className="text-gray-700">{recording.generatedPlan.summary}</p>
                      <div className="mt-2 flex items-center gap-2 text-sm text-gray-600">
                        <span>Confidence: {Math.round(recording.generatedPlan.confidence * 100)}%</span>
                        <span>•</span>
                        <span>{recording.generatedPlan.steps.length} steps</span>
                      </div>
                    </div>

                    <div>
                      <h4 className="font-medium mb-2">Steps</h4>
                      <div className="space-y-2">
                        {recording.generatedPlan.steps.slice(0, 3).map((step) => (
                          <div key={step.number} className="flex items-center gap-2 text-sm">
                            <Badge variant="outline">{step.number}</Badge>
                            <span className="flex-1">{step.title}</span>
                            <span className="text-gray-500">{step.duration}</span>
                          </div>
                        ))}
                        {recording.generatedPlan.steps.length > 3 && (
                          <div className="text-sm text-gray-500">
                            ... and {recording.generatedPlan.steps.length - 3} more steps
                          </div>
                        )}
                      </div>
                    </div>

                    <div className="flex gap-2 pt-2">
                      <Link href={`/feedback/${recording.id}`}>
                        <Button size="sm" className="flex items-center gap-2">
                          <MessageSquare className="h-4 w-4" />
                          Provide Feedback
                        </Button>
                      </Link>
                      <Button 
                        size="sm" 
                        variant="outline" 
                        className="flex items-center gap-2"
                        onClick={() => {
                          // Open detailed view in new tab
                          const detailWindow = window.open('', '_blank');
                          if (detailWindow) {
                            detailWindow.document.write(`
                              <html>
                                <head><title>Recording Details</title></head>
                                <body style="font-family: system-ui; padding: 20px;">
                                  <h1>Recording Details</h1>
                                  <h2>Task: ${recording.taskDescription}</h2>
                                  <h3>Full Plan:</h3>
                                  <pre style="background: #f5f5f5; padding: 15px; border-radius: 5px; overflow: auto;">
${JSON.stringify(recording.generatedPlan, null, 2)}
                                  </pre>
                                </body>
                              </html>
                            `);
                          }
                        }}
                      >
                        <ExternalLink className="h-4 w-4" />
                        View Details
                      </Button>
                    </div>
                  </div>
                ) : (
                  <div className="text-gray-600">
                    {recording.status === "processing" ? "Processing..." : "No plan generated"}
                  </div>
                )}
              </CardContent>
            </Card>
          ))}
        </div>
      )}
    </div>
  );
}