// ActivityRecorder Background Service Worker
console.log('ActivityRecorder background script loaded');

// Configuration - Use localhost proxy to bypass CSP restrictions
const API_BASE_URL = 'http://localhost:8080';

// State management
let isRecording = false;
let currentSessionId = null;
let recordingStartTime = null;
let capturedActivities = [];

// Track tab navigation events (works even on CSP-restricted sites)
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (isRecording && changeInfo.status === 'complete' && tab.url) {
    // Only capture navigation events, not every tab update
    if (changeInfo.url || (changeInfo.status === 'complete' && tab.url)) {
      console.log('🌐 NAVIGATION DETECTED:', tab.url);
      
      const navigationActivity = {
        action: 'navigation',
        url: tab.url,
        title: tab.title || 'Unknown Page',
        timestamp: Date.now(),
        tabId: tabId
      };
      
      // Store navigation activity with screenshot (with delay to ensure page is ready)
      capturedActivities.push(navigationActivity);
      
      // Add a small delay to ensure the page is fully loaded before screenshot
      setTimeout(() => {
        captureActivityWithScreenshot(navigationActivity, tabId);
      }, 500);
      
      console.log('📊 Navigation captured:', tab.url);
    }
  }
});

// Get API URL
function getApiUrl() {
  return API_BASE_URL;
}

// Session management
function startRecording() {
  console.log('Starting recording session');
  isRecording = true;
  currentSessionId = 'session_' + Date.now();
  recordingStartTime = Date.now();
  capturedActivities = [];
  
  // Store session state
  chrome.storage.local.set({
    isRecording: true,
    sessionId: currentSessionId,
    startTime: recordingStartTime
  });
  
  // Update extension badge
  chrome.action.setBadgeText({ text: 'REC' });
  chrome.action.setBadgeBackgroundColor({ color: '#ff4444' });
  
  console.log('🔴 Recording started successfully - sessionId:', currentSessionId);
  
  // Notify all content scripts to start monitoring
  chrome.tabs.query({}, (tabs) => {
    tabs.forEach(tab => {
      chrome.tabs.sendMessage(tab.id, { type: 'START_MONITORING' }, (response) => {
        if (chrome.runtime.lastError) {
          console.warn('Could not notify tab', tab.id, ':', chrome.runtime.lastError.message);
        } else {
          console.log('✅ Notified tab', tab.id, 'to start monitoring');
        }
      });
    });
  });
}

function stopRecording() {
  console.log('Stopping recording session');
  isRecording = false;
  
  // Store final session data
  chrome.storage.local.set({
    isRecording: false,
    capturedActivities: capturedActivities,
    endTime: Date.now()
  });
  
  // Clear badge
  chrome.action.setBadgeText({ text: '' });
  
  return {
    sessionId: currentSessionId,
    startTime: recordingStartTime,
    endTime: Date.now(),
    activities: capturedActivities
  };
}

// Rate limiting for screenshot capture
let lastScreenshotTime = 0;
const SCREENSHOT_RATE_LIMIT_MS = 2000; // 2 seconds between screenshots
let screenshotQueue = [];
let isProcessingScreenshots = false;

// Activity capture and processing with enhanced rate limiting
async function captureActivityWithScreenshot(activity, tabId) {
  try {
    console.log('🔍 CAPTURE START - Activity:', activity.action, 'TabId:', tabId);
    
    // Check rate limit for screenshots
    const now = Date.now();
    const timeSinceLastScreenshot = now - lastScreenshotTime;
    let screenshotDataUrl = null;
    
    if (timeSinceLastScreenshot >= SCREENSHOT_RATE_LIMIT_MS) {
      try {
        // Try to get the specific tab first, then fall back to active tab
        let targetTab = null;
        try {
          if (tabId) {
            targetTab = await chrome.tabs.get(tabId);
            console.log('🎯 Target tab found:', targetTab?.id, 'URL:', targetTab?.url, 'Status:', targetTab?.status);
          }
        } catch (tabError) {
          console.warn('⚠️ Could not get specific tab, falling back to active tab');
        }
        
        // If no specific tab, get active tab
        if (!targetTab) {
          const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
          targetTab = activeTab;
          console.log('🔍 Active tab found:', targetTab?.id, 'URL:', targetTab?.url, 'Status:', targetTab?.status);
        }
        
        // Check if tab is in a valid state for screenshot capture
        if (targetTab && targetTab.id && targetTab.status === 'complete' && 
            !targetTab.url?.startsWith('chrome://') && 
            !targetTab.url?.startsWith('chrome-extension://') &&
            !targetTab.url?.startsWith('edge://') &&
            !targetTab.url?.startsWith('about:') &&
            !targetTab.url?.startsWith('moz-extension://')) {
          
          console.log('📸 Attempting screenshot capture for tab:', targetTab.id, 'in window:', targetTab.windowId);
          
          // Screenshot capture with host_permissions (all_urls)
          try {
            screenshotDataUrl = await chrome.tabs.captureVisibleTab(targetTab.windowId, {
              format: 'jpeg',
              quality: 50
            });
            lastScreenshotTime = now;
            console.log('✅ Screenshot captured successfully! Size:', screenshotDataUrl?.length || 'N/A');
            console.log('✅ Screenshot data starts with:', screenshotDataUrl?.substring(0, 50) + '...');
          } catch (captureError) {
            console.error('❌ Screenshot capture error:', captureError.message);
            
            if (captureError.message?.includes('MAX_CAPTURE_VISIBLE_TAB_CALLS_PER_SECOND')) {
              console.warn('⚠️ Screenshot rate limit exceeded - skipping');
            } else if (captureError.message?.includes('cannot be edited') || 
                       captureError.message?.includes('dragging') ||
                       captureError.message?.includes('not permitted')) {
              console.warn('⚠️ Screenshot skipped - tab in invalid state');
            } else {
              console.error('❌ Unexpected screenshot error. Check manifest host_permissions.');
            }
          }
        } else {
          const reason = !targetTab ? 'no target tab' : 
                        targetTab.status !== 'complete' ? `tab status: ${targetTab.status}` :
                        targetTab.url?.startsWith('chrome') || targetTab.url?.startsWith('moz-extension') ? 'system page' : 'unknown reason';
          console.warn('❌ Screenshot skipped:', reason, 'Tab:', targetTab?.url);
        }
      } catch (screenshotError) {
        console.error('❌ Screenshot capture failed with error:', screenshotError.message);
      }
    } else {
      const waitTime = SCREENSHOT_RATE_LIMIT_MS - timeSinceLastScreenshot;
      console.log(`⏳ Screenshot rate limited - waiting ${waitTime}ms before next capture`);
    }
    
    // Upload screenshot and get reference if successful
    let screenshotRef = null;
    if (screenshotDataUrl) {
      try {
        const screenshotResponse = await fetch(`${getApiUrl()}/api/screenshots/upload`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            screenshot: screenshotDataUrl,
            sessionId: currentSessionId,
            timestamp: Date.now()
          })
        });
        
        if (screenshotResponse.ok) {
          const screenshotData = await screenshotResponse.json();
          screenshotRef = screenshotData.screenshotId;
          console.log('✅ Screenshot uploaded successfully, ref:', screenshotRef);
          console.log('✅ Screenshot ID length:', screenshotRef?.length);
        } else {
          console.error('❌ Screenshot upload failed:', screenshotResponse.status, screenshotResponse.statusText);
          const errorText = await screenshotResponse.text();
          console.error('❌ Error details:', errorText);
        }
      } catch (uploadError) {
        console.warn('Screenshot upload failed:', uploadError);
      }
    }
    
    // Create enhanced activity object
    const enhancedActivity = {
      ...activity,
      timestamp: Date.now(),
      sessionId: currentSessionId,
      tabId: tabId,
      screenshot: screenshotRef, // Reference ID instead of full data
      captureTime: new Date().toISOString()
    };
    
    console.log('🔍 Enhanced activity created with screenshot:', screenshotRef);
    console.log('🔍 Activity object:', JSON.stringify({
      action: enhancedActivity.action,
      screenshot: enhancedActivity.screenshot,
      url: enhancedActivity.url
    }));
    
    // Store activity locally
    capturedActivities.push(enhancedActivity);
    
    // Stream activity to backend API
    try {
      const streamPayload = {
        sessionId: currentSessionId,
        activity: enhancedActivity
      };
      
      await fetch(`${getApiUrl()}/api/stream-activity`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(streamPayload)
      });
    } catch (streamError) {
      console.warn('Activity streaming failed:', streamError);
    }
    
    console.log('📊 Activity captured:', enhancedActivity.action, 'on', enhancedActivity.url);
    
  } catch (error) {
    console.error('Error capturing activity with screenshot:', error);
  }
}

// Helper function to stream activity to backend
async function streamActivityToBackend(activity) {
  try {
    const streamPayload = {
      sessionId: currentSessionId,
      activity: activity
    };
    
    await fetch(`${getApiUrl()}/api/stream-activity`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(streamPayload)
    });
  } catch (streamError) {
    console.warn('Activity streaming failed:', streamError);
  }
}

// Message handling from content scripts and popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('📨 Background received message:', message.type);
  
  if (message.type === 'ACTIVITY_CAPTURED') {
    console.log('📨 ACTIVITY_CAPTURED message received');
    console.log('📨 Is recording:', isRecording);
    console.log('📨 Sender tab:', sender.tab?.id);
    console.log('📨 Activity data:', message.activity);
    
    if (isRecording && sender.tab) {
      captureActivityWithScreenshot(message.activity, sender.tab.id);
      console.log('📨 Activity passed to captureActivityWithScreenshot');
    } else {
      console.warn('📨 Activity NOT processed - isRecording:', isRecording, 'senderTab:', !!sender.tab);
    }
    sendResponse({ success: true });
    return true; // Keep message channel open for async response
  }
  
  if (message.type === 'START_RECORDING') {
    startRecording();
    sendResponse({ success: true, sessionId: currentSessionId });
    return true;
  }
  
  if (message.type === 'STOP_RECORDING') {
    const sessionData = stopRecording();
    sendResponse({ success: true, sessionData });
    return true;
  }
  
  if (message.type === 'GET_STATUS') {
    sendResponse({
      isRecording,
      sessionId: currentSessionId,
      activityCount: capturedActivities.length
    });
    return true;
  }
  
  if (message.type === 'GENERATE_PLAN') {
    generatePlan(message.taskDescription).then(result => {
      sendResponse(result);
    }).catch(error => {
      sendResponse({ success: false, error: error.message });
    });
    return true; // Keep message channel open for async response
  }
  
  if (message.type === 'CAPTURE_SCREENSHOT') {
    // Handle screenshot capture request from content script
    if (sender.tab && sender.tab.id) {
      chrome.tabs.captureVisibleTab(null, { format: 'png' }, (dataUrl) => {
        if (chrome.runtime.lastError) {
          console.error('Screenshot capture error:', chrome.runtime.lastError);
          sendResponse({ success: false, error: chrome.runtime.lastError.message });
        } else {
          sendResponse({ success: true, screenshot: dataUrl });
        }
      });
      return true; // Keep message channel open for async response
    }
  }
});

// Plan generation
async function generatePlan(taskDescription) {
  try {
    console.log('🔍 PLAN GENERATION START');
    console.log('🔍 Current session ID:', currentSessionId);
    console.log('🔍 Captured activities count:', capturedActivities.length);
    console.log('🔍 Captured activities:', capturedActivities.map(a => `${a.action} on ${a.url}`));
    
    if (!currentSessionId || capturedActivities.length === 0) {
      console.error('❌ PLAN GENERATION FAILED: No recorded activities found');
      console.error('❌ Session ID:', currentSessionId);
      console.error('❌ Activities count:', capturedActivities.length);
      throw new Error('No recorded activities found');
    }
    
    const sessionData = {
      sessionId: currentSessionId,
      taskDescription,
      startTime: recordingStartTime,
      endTime: Date.now(),
      activities: capturedActivities
    };
    
    console.log('🤖 Generating plan for', capturedActivities.length, 'activities');
    
    const response = await fetch(`${getApiUrl()}/api/generate-plan`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(sessionData)
    });
    
    if (!response.ok) {
      throw new Error(`API Error: ${response.status} ${response.statusText}`);
    }
    
    const result = await response.json();
    return result;
    
  } catch (error) {
    console.error('Plan generation failed:', error);
    throw error;
  }
}

// Tab management and context tracking
chrome.tabs.onActivated.addListener(async (activeInfo) => {
  if (!isRecording) return;
  
  try {
    const tab = await chrome.tabs.get(activeInfo.tabId);
    
    // Only capture if tab is in a valid state and not a system page
    if (tab && tab.url && tab.status === 'complete' &&
        !tab.url.startsWith('chrome://') && 
        !tab.url.startsWith('chrome-extension://') &&
        !tab.url.startsWith('moz-extension://') &&
        !tab.url.startsWith('about:')) {
      
      // Add delay to reduce rate limit issues
      setTimeout(async () => {
        try {
          const activity = {
            action: 'tab_focus',
            url: tab.url,
            title: tab.title,
            tabId: activeInfo.tabId
          };
          
          await captureActivityWithScreenshot(activity, activeInfo.tabId);
        } catch (error) {
          console.error('Error capturing delayed tab focus:', error);
        }
      }, 500);
    } else {
      console.log('⏭️ Skipping tab focus - invalid tab:', tab?.status, tab?.url);
    }
  } catch (error) {
    console.error('Error capturing tab activation:', error);
  }
});

chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  if (!isRecording) return;
  
  // Only capture when tab is completely loaded and has a valid URL
  if (changeInfo.status === 'complete' && tab.url && 
      !tab.url.startsWith('chrome://') && 
      !tab.url.startsWith('chrome-extension://') &&
      !tab.url.startsWith('moz-extension://') &&
      !tab.url.startsWith('about:')) {
    
    // Add a longer delay to ensure the page is fully ready and avoid rate limits
    setTimeout(async () => {
      try {
        const activity = {
          action: 'navigation',
          url: tab.url,
          title: tab.title,
          tabId: tabId
        };
        
        await captureActivityWithScreenshot(activity, tabId);
      } catch (error) {
        console.error('Error capturing navigation:', error);
      }
    }, 1000); // Increased delay to reduce rate limit issues
  }
});

// Extension lifecycle
chrome.runtime.onInstalled.addListener(() => {
  console.log('🎉 ActivityRecorder extension installed');
  
  // Initialize storage
  chrome.storage.local.set({
    isRecording: false,
    sessionId: null,
    capturedActivities: []
  });
});

// Cleanup on extension suspension
chrome.runtime.onSuspend.addListener(() => {
  if (isRecording) {
    stopRecording();
  }
});