// ActivityRecorder Content Script - Fixed Version
// Prevents extension context invalidation errors

console.log('🔧 ActivityRecorder content script loading...');

// Prevent multiple injections
if (window.activityRecorderInjected) {
  console.log('⚠️ ActivityRecorder already injected, skipping');
} else {
  window.activityRecorderInjected = true;

  // Extension state tracking
  let isMonitoring = false;
  let extensionContextValid = true;
  let activityBuffer = [];

  // Context validation function
  function validateExtensionContext() {
    try {
      if (!chrome.runtime || !chrome.runtime.id) {
        if (extensionContextValid) {
          console.warn('⚠️ Extension context invalidated - stopping all activity capture');
          extensionContextValid = false;
          isMonitoring = false;
        }
        return false;
      }
      return true;
    } catch (error) {
      if (extensionContextValid) {
        console.warn('⚠️ Extension context error - stopping monitoring:', error.message);
        extensionContextValid = false;
        isMonitoring = false;
      }
      return false;
    }
  }

  // Safe activity capture that checks context validity
  function safeCaptureActivity(activity) {
    // Quick exit if we know context is invalid
    if (!extensionContextValid || !isMonitoring) return;
    
    // Validate context before attempting to send
    if (!validateExtensionContext()) return;
    
    // Add timestamp and metadata
    activity.timestamp = Date.now();
    activity.userAgent = navigator.userAgent;
    activity.viewport = {
      width: window.innerWidth,
      height: window.innerHeight
    };
    
    try {
      chrome.runtime.sendMessage({
        type: 'ACTIVITY_CAPTURED',
        activity: activity
      }, (response) => {
        if (chrome.runtime.lastError) {
          const errorMsg = chrome.runtime.lastError.message;
          if (errorMsg?.includes('Extension context invalidated') || 
              errorMsg?.includes('context invalidated')) {
            console.warn('⚠️ Context invalidated during send - stopping monitoring');
            extensionContextValid = false;
            isMonitoring = false;
          }
        }
      });
    } catch (error) {
      if (error.message.includes('context') || error.message.includes('Extension')) {
        extensionContextValid = false;
        isMonitoring = false;
      }
    }
  }

  // Event handlers with context validation
  function handleClick(event) {
    if (!isMonitoring || !extensionContextValid) return;
    
    const element = event.target;
    const activity = {
      action: 'click',
      url: window.location.href,
      title: document.title,
      element: {
        tagName: element.tagName,
        id: element.id || null,
        className: element.className || null,
        textContent: element.textContent?.substring(0, 100) || null,
        href: element.href || null
      },
      coordinates: {
        x: event.clientX,
        y: event.clientY
      }
    };
    
    safeCaptureActivity(activity);
  }

  function handleScroll(event) {
    if (!isMonitoring || !extensionContextValid) return;
    
    // Throttle scroll events
    if (window.lastScrollCapture && Date.now() - window.lastScrollCapture < 3000) {
      return;
    }
    window.lastScrollCapture = Date.now();
    
    const activity = {
      action: 'scroll',
      url: window.location.href,
      title: document.title,
      scrollPosition: {
        x: window.scrollX,
        y: window.scrollY
      },
      documentHeight: document.documentElement.scrollHeight,
      viewportHeight: window.innerHeight
    };
    
    safeCaptureActivity(activity);
  }

  function handleInput(event) {
    if (!isMonitoring || !extensionContextValid) return;
    
    // Throttle input events
    if (event.target.dataset.lastInputCapture && 
        Date.now() - parseInt(event.target.dataset.lastInputCapture) < 2000) {
      return;
    }
    event.target.dataset.lastInputCapture = Date.now().toString();
    
    const element = event.target;
    
    // Skip sensitive inputs
    if (element.type === 'password' || 
        element.type === 'hidden' ||
        element.name?.toLowerCase().includes('password')) {
      return;
    }
    
    const activity = {
      action: 'input',
      url: window.location.href,
      title: document.title,
      element: {
        tagName: element.tagName,
        type: element.type || null,
        id: element.id || null,
        name: element.name || null,
        placeholder: element.placeholder || null,
        valueLength: element.value?.length || 0
      }
    };
    
    safeCaptureActivity(activity);
  }

  function handleFormSubmit(event) {
    if (!isMonitoring || !extensionContextValid) return;
    
    const form = event.target;
    const formData = {};
    
    // Safely capture form data (excluding sensitive fields)
    try {
      const formDataObj = new FormData(form);
      for (let [key, value] of formDataObj.entries()) {
        // Skip sensitive fields
        if (!key.toLowerCase().includes('password') && 
            !key.toLowerCase().includes('ssn') &&
            !key.toLowerCase().includes('card')) {
          formData[key] = typeof value === 'string' ? value.substring(0, 100) : '[File]';
        }
      }
    } catch (error) {
      console.warn('Error capturing form data:', error);
    }
    
    const activity = {
      action: 'form_submit',
      url: window.location.href,
      title: document.title,
      element: {
        tagName: form.tagName,
        id: form.id || null,
        className: form.className || null,
        action: form.action || null,
        method: form.method || 'GET'
      },
      formData: formData
    };
    
    safeCaptureActivity(activity);
  }

  function handlePageUnload(event) {
    if (!isMonitoring || !extensionContextValid) return;
    
    const activity = {
      action: 'page_unload',
      url: window.location.href,
      title: document.title,
      timeOnPage: Date.now() - (window.pageLoadTime || Date.now())
    };
    
    safeCaptureActivity(activity);
  }

  // Monitoring control functions
  function startMonitoring() {
    if (isMonitoring || !validateExtensionContext()) return;
    
    console.log('📊 Starting activity monitoring on:', window.location.href);
    isMonitoring = true;
    
    // Capture initial page load activity
    safeCaptureActivity({
      action: 'page_load',
      url: window.location.href,
      title: document.title,
      referrer: document.referrer || null
    });

    // Add event listeners
    document.addEventListener('click', handleClick, true);
    document.addEventListener('submit', handleFormSubmit, true);
    document.addEventListener('input', handleInput, true);
    document.addEventListener('scroll', handleScroll, true);
    window.addEventListener('beforeunload', handlePageUnload, true);
  }

  function stopMonitoring() {
    if (!isMonitoring) return;
    
    console.log('⏹️ Stopping activity monitoring');
    isMonitoring = false;
    
    // Remove event listeners
    document.removeEventListener('click', handleClick, true);
    document.removeEventListener('submit', handleFormSubmit, true);
    document.removeEventListener('input', handleInput, true);
    document.removeEventListener('scroll', handleScroll, true);
    window.removeEventListener('beforeunload', handlePageUnload, true);
  }

  // Initial recording status check
  try {
    if (validateExtensionContext()) {
      chrome.runtime.sendMessage({ type: 'GET_RECORDING_STATUS' }, (response) => {
        if (chrome.runtime.lastError) {
          console.warn('Failed to get recording status:', chrome.runtime.lastError.message);
          return;
        }
        
        if (response?.isRecording) {
          startMonitoring();
        }
      });
    }
  } catch (error) {
    console.warn('Failed to check recording status:', error);
  }

  // Message listener with enhanced error handling
  try {
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      // Validate context before processing any messages
      if (!validateExtensionContext()) {
        return;
      }
      
      try {
        if (message.type === 'START_MONITORING') {
          startMonitoring();
          sendResponse({ success: true });
        } else if (message.type === 'STOP_MONITORING') {
          stopMonitoring();
          sendResponse({ success: true });
        }
      } catch (contextError) {
        console.warn('⚠️ Context error in message handler:', contextError.message);
        extensionContextValid = false;
        isMonitoring = false;
      }
    });
  } catch (error) {
    console.warn('Failed to set up message listener:', error);
    extensionContextValid = false;
  }

  // Monitor for SPA navigation
  window.pageLoadTime = Date.now();
  let lastUrl = window.location.href;
  
  const observer = new MutationObserver(() => {
    if (!isMonitoring || !extensionContextValid) return;
    
    if (lastUrl !== window.location.href) {
      lastUrl = window.location.href;
      safeCaptureActivity({
        action: 'spa_navigation',
        url: window.location.href,
        title: document.title
      });
    }
  });
  
  try {
    observer.observe(document.body, {
      childList: true,
      subtree: true
    });
  } catch (error) {
    console.warn('Failed to set up mutation observer:', error);
  }

  console.log('✅ ActivityRecorder content script initialized with context validation');
}